(*-------------------------------------------------------------------------------------------*)
(* 0 : TABLES DE TRANSITIONS *)
(*-------------------------------------------------------------------------------------------*)
(* Construction d'une table de transitions pour un groupe fini de cardinal 'n' d'élément neutre 'e' *)
(* de produit '$' dont les éléments sont encodés/décodés sous forme d'entiers de 0 à n-1 par les fonctions *)
(* 'encode' et 'decode', pour une liste 'lg' de générateurs du groupe représentés dans la table *)
(* par des caractères attribués par la fonction 'chr' : cette table 't' a pour longueur 'n' et *)
(* si t.(i)=c alors, si l'élément <i> du groupe représenté par i est  différent du neutre et si g est *)
(* le générateur du groupe représenté par le caractère c, l'élément <i>g'du groupe où g' est l'inverse de g
 est en nombre de générateurs à distance du neutre inférieure d'une unité à celle de <i> *)

(* La fonction 'transitions' est éventuellement utilisée en 7 : MOUVEMENTS 2, ENCODAGE ET DÉCODAGE *)
(* avec une liste de quarts de tours et une liste de demi-tours ou quarts de tours des faces *)
(* gauche, basse et postérieure du cube (le minicube antérieur-haut-droit étant considéré comme fixe) : *)
(* si un des deux fichiers de table de transitions  n'est pas trouvé il est recalculé *)
(* Durée totale approximative de calcul des deux tables sur MacBook Pro M1: environ 45 secondes *)
(* dont 17 secondes pour la table des quarts de tours *)

let transitions (n: int) (e: ' a) ((prefix $): ' a -> ' a -> ' a) (encode: ' a -> int)
(decode: int -> ' a) (lg: ' a list) (chr: ' a -> char) =
	let t = make_vect n ` ` in
		t.(encode e) <- `$`;
		let suivants i1 t1 i2 t2 =
			i2 := 0;
			let ajouter y c =
				if t.(y) = ` ` then (t.(y) <- c; t2.(!i2) <- y; incr i2)
			in
				let f y = let x = decode y in
						do_list (fun gen -> ajouter (encode (x $ gen)) (chr gen)) lg
				in
					do_vect f (sub_vect t1 0 !i1)
		in
			let total = ref 1 in
				let rec aux i1 t1 i2 t2 i =
					let tm = sys__time () in
						suivants i1 t1 i2 t2;
						if !i2 > 0 then (
								total := !total + !i2;
								printf__printf "à distance %d, %d éléments ajoutés en %0.4f secondes" i !i2 (sys__time () -. tm);
								print_newline ();
								aux i2 t2 i1 t1 (i + 1)
							)
				in
					let i1 = ref 1 and i2 = ref 0
					and t1 = make_vect n 0
					and t2 = make_vect n 0
					in
						t1.(0) <- encode e;
						printf__printf "Au départ 1 seul élément : le neutre";
						print_newline ();
						let tm = sys__time () in
							aux i1 t1 i2 t2 1;
							printf__printf "\nEn tout %d éléments obtenus en %0.4f secondes\n" !total (sys__time () -. tm);
							print_newline ();
							t
;;

(*-------------------------------------------------------------------------------------------*)
(* 1 : TYPES *)
(*-------------------------------------------------------------------------------------------*)

type mouvement1 =
	{mutable mv1: (int vect * int vect vect) list}
;;

type context =
	{mutable matrice: int vect vect}
;;

type repere = {mutable plan: int * int * int * int};;

type ops = OPS of (unit -> unit) * (unit -> unit) * (unit -> unit);;

type couleur = ORANGE | VERT | BLANC | ROUGE | BLEU | JAUNE | GRIS;;

type bouton = {titre: string;
		orx: int; ory: int;
		largeur: int; hauteur: int;
		mutable couleur: couleur;
		mutable action: unit -> unit;
		mutable bas: bool}
;;

type cube1 =
	{
		anime1: bool ref;
		mutable mouvement1: mouvement1;
		mutable mvi: mouvement1;
		mutable context1: context;
		mutable repere1: repere;
		mutable rotations_cube1: ops * ops;
		mutable rotations_faces1: ops * ops * ops * ops;
		mutable boutons1: bouton vect;
		mutable boutons11: bouton vect
	}
;;

type couple = COUPLE of (int vect * int vect) | NIL;;

(*-------------------------------------------------------------------------------------------*)
(* 2 : DIVERS *)
(*-------------------------------------------------------------------------------------------*)

(* matrices en dimension 3 *)
let matrice_nulle = [|[|0; 0; 0|]; [|0; 0; 0|]; [|0; 0; 0|]|];;

let vect v = if vect_length v = 3 then (v.(0), v.(1), v.(2))
	else failwith "vect"
;;

let matscal a = let m = make_matrix 3 3 0 in
		for i = 0 to 2 do
			m.(i).(i) <- a
		done;
		m
;;

let id = matscal 1 and idm = matscal (- 1);;

(* produit du vecteur ligne entier v par la matrice entière m *)
let prefix /:/ v m =
	let w j = let t = ref 0 in for k = 0 to vect_length v - 1 do
				t := !t + m.(k).(j) * v.(k) done;
			!t in
		[|w 0; w 1; w 2|]
;;

(*produit du scalaire a par la matrice m*)
let prefix /../ a m =
	map_vect (fun x -> map_vect (fun t -> a * t) x) m;;

(* produit matriciel *)
let prefix /./ m m1 = map_vect (fun v -> v /:/ m1) m;;

(* somme matricielle *)
let prefix /+/ m1 m2 =
	let m = make_matrix 3 3 0 in
		for i = 0 to 2 do
			for j = 0 to 2 do
				m.(i).(j) <- m1.(i).(j) + m2.(i).(j)
			done;
		done;
		m
;;

(* matrice diagonale *)
let diag a b c = [|[|a; 0; 0|]; [|0; b; 0|]; [|0; 0; c|]|];;

(* transposée de la matrice m  qui en est aussi l'inverse : *)
(* quand m est orthogonale *)
let transpose m =
	let m1 = make_matrix 3 3 0 in
		for i = 0 to 2 do
			for j = 0 to 2 do
				m1.(j).(i) <- m.(i).(j)
			done;
		done;
		m1
;;

(* produit scalaire *)
let prefix /|/ v w = v.(0) * w.(0) + v.(1) * w.(1) + v.(2) * w.(2);;

(* matrices des rotations d'un quart de tour autour des axes : *)
(* (opèrent à droite sur les lignes) *)

(* sens des aiguilles d'une montre *)
let rot v = match list_of_vect v with
		| [1; 0; 0] -> [|[|1; 0; 0|]; [|0; 0; - 1|]; [|0; 1; 0|]|]
		| [0; 1; 0] -> [|[|0; 0; 1|]; [|0; 1; 0|]; [|- 1; 0; 0|]|]
		| [0; 0; 1] -> [|[|0; - 1; 0|]; [|1; 0; 0|]; [|0; 0; 1|]|]
		| [- 1; 0; 0] -> [|[|1; 0; 0|]; [|0; 0; 1|]; [|0; - 1; 0|]|]
		| [0; - 1; 0] -> [|[|0; 0; - 1|]; [|0; 1; 0|]; [|1; 0; 0|]|]
		| [0; 0; - 1] -> [|[|0; 1; 0|]; [|- 1; 0; 0|]; [|0; 0; 1|]|]
		| _ -> failwith "rot"
;;

(* sens inverse des aiguilles d'une montre *)
let rot' v = match list_of_vect v with
		| [1; 0; 0] -> [|[|1; 0; 0|]; [|0; 0; 1|]; [|0; - 1; 0|]|]
		| [0; 1; 0] -> [|[|0; 0; - 1|]; [|0; 1; 0|]; [|1; 0; 0|]|]
		| [0; 0; 1] -> [|[|0; 1; 0|]; [|- 1; 0; 0|]; [|0; 0; 1|]|]
		| [- 1; 0; 0] -> [|[|1; 0; 0|]; [|0; 0; - 1|]; [|0; 1; 0|]|]
		| [0; - 1; 0] -> [|[|0; 0; 1|]; [|0; 1; 0|]; [|- 1; 0; 0|]|]
		| [0; 0; - 1] -> [|[|0; - 1; 0|]; [|1; 0; 0|]; [|0; 0; 1|]|]
		| _ -> failwith "rot'"
;;

(* liste dans l'ordre des éléments de l satisfaisant 'critère' *)
let rec select critere l = match l with
		t :: r -> let l1 = select critere r in if critere t then t :: l1 else l1
		| _ -> []
;;

(* liste des entiers de 0 à n - 1 *)
let liste n =
	let v = make_vect n 0 in
		for i = 0 to n - 1 do
			v.(i) <- i
		done;
		list_of_vect v
;;

(* permutation aléatoire des éléments d'une liste l *)
let random_list l =
	let n = list_length l and l1 = ref []
	in
		for i = 0 to n - 1 do
			l1 := (vect_of_list (subtract l !l1)).(random__int (n - i)) :: !l1
		done;
		!l1
;;

(* signature de la permutation p des éléments de la liste l *)
let sign l p =
	let n = list_length l and v = vect_of_list l
	and m = ref 1 in
		for i = 0 to n - 1 do
			for j = i + 1 to n - 1 do
				let a = v.(i) and b = v.(j) in
					if p a > p b && b > a || p b > p a && a > b then m := - !m;
			done;
		done;
		!m
;;

(* exécution d'une liste de mouvements *)
let rec exe1 l = match l with
		t :: r -> t (); exe1 r;
		| [] -> ()
;;
(*
let rec op_names_from_string s =
	let (t, r) = scinde s in
		if r = "" then [t] else t :: op_names_from_string r
;;
*)
let rec op_names_from_string s =
	let tete s =
		let l = string_length s in
			if l = 0 then ""
			else if l = 1 then sub_string s 0 1
			else if l = 2 then if s.[1] = `0` || s.[1] = `'` then s else sub_string s 0 1
			else match s.[1], s.[2] with
				| `0`, `'` -> sub_string s 0 3
				| `0`, _ -> sub_string s 0 2
				| `'`, _ -> sub_string s 0 2
				| _ -> sub_string s 0 1
	in
		let scinde s =
			let t = tete s and ls = string_length s in
				let lt = string_length t in
					let r = sub_string s lt (ls - lt) in
						(t, r)
		in
			let (t, r) = scinde s in
				if r = "" then [t] else t :: op_names_from_string r
;;

let format_string op_names_string = "exec \"" ^ op_names_string ^ "\";;\n";;

(*-------------------------------------------------------------------------------------------*)
(* 3 : INDICES *)
(*-------------------------------------------------------------------------------------------*)

(* indices des coins *)
(*let indices = let l = ref [] in
		for k = 1 downto - 1 do
			for j = 1 downto - 1 do
				for i = 1 downto - 1 do l := [|i; j; k|] :: !l
				done
			done
		done;
		select (fun x -> x.(0) * x.(1) * x.(2) <> 0) !l
		(*sort__sort (fun x y -> x > y) (select (fun x -> x.(0) * x.(1) * x.(2) <> 0) !l)*)

;;
*)
let indices = [[|1; 1; 1|]; [|1; 1; - 1|]; [|1; - 1; 1|]; [|1; - 1; - 1|]; [|- 1; 1; 1|];
		[|- 1; 1; - 1|]; [|- 1; - 1; 1|]; [|- 1; - 1; - 1|]];;
(* liste des coins : ici il n'y a que des coins! *)
let coins = indices;;


(*-------------------------------------------------------------------------------------------*)
(* 4 : GROUPE DES ROTATIONS DU CUBE *)
(*-------------------------------------------------------------------------------------------*)

let groupe_du_cube =
	[
		[|[|1; 0; 0|]; [|0; 1; 0|]; [|0; 0; 1|]|];
		[|[|1; 0; 0|]; [|0; - 1; 0|]; [|0; 0; - 1|]|];
		[|[|- 1; 0; 0|]; [|0; 1; 0|]; [|0; 0; - 1|]|];
		[|[|- 1; 0; 0|]; [|0; - 1; 0|]; [|0; 0; 1|]|];
		[|[|1; 0; 0|]; [|0; 0; - 1|]; [|0; 1; 0|]|];
		[|[|0; 0; 1|]; [|0; 1; 0|]; [|- 1; 0; 0|]|];
		[|[|0; - 1; 0|]; [|1; 0; 0|]; [|0; 0; 1|]|];
		[|[|1; 0; 0|]; [|0; 0; 1|]; [|0; - 1; 0|]|];
		[|[|0; 0; - 1|]; [|0; 1; 0|]; [|1; 0; 0|]|];
		[|[|0; 1; 0|]; [|- 1; 0; 0|]; [|0; 0; 1|]|];
		[|[|0; - 1; 0|]; [|- 1; 0; 0|]; [|0; 0; - 1|]|];
		[|[|0; 1; 0|]; [|1; 0; 0|]; [|0; 0; - 1|]|];
		[|[|- 1; 0; 0|]; [|0; 0; - 1|]; [|0; - 1; 0|]|];
		[|[|0; 0; - 1|]; [|0; - 1; 0|]; [|- 1; 0; 0|]|];
		[|[|0; 0; 1|]; [|0; - 1; 0|]; [|1; 0; 0|]|];
		[|[|- 1; 0; 0|]; [|0; 0; 1|]; [|0; 1; 0|]|];
		[|[|0; 1; 0|]; [|0; 0; 1|]; [|1; 0; 0|]|];
		[|[|0; 0; - 1|]; [|- 1; 0; 0|]; [|0; 1; 0|]|];
		[|[|0; 0; 1|]; [|- 1; 0; 0|]; [|0; - 1; 0|]|];
		[|[|0; 1; 0|]; [|0; 0; - 1|]; [|- 1; 0; 0|]|];
		[|[|0; 0; - 1|]; [|1; 0; 0|]; [|0; - 1; 0|]|];
		[|[|0; - 1; 0|]; [|0; 0; - 1|]; [|1; 0; 0|]|];
		[|[|0; - 1; 0|]; [|0; 0; 1|]; [|- 1; 0; 0|]|];
		[|[|0; 0; 1|]; [|1; 0; 0|]; [|0; 1; 0|]|]
	]
;;

(* représentation des (inverses des) éléments du groupe par les éléments a0, d0 ,h0 *)
let decomposition r =
	let lops = [""; "a0a0"; "d0d0"; "h0h0"; "a0'"; "d0'"; "h0'"; "a0"; "d0"; "h0";
			"h0'a0a0"; "h0a0a0"; "d0d0a0"; "d0a0a0"; "d0'a0a0"; "d0d0a0'"; "h0a0";
			"d0a0'"; "d0'a0"; "h0a0'"; "d0a0"; "h0'a0'"; "h0'a0"; "d0'a0'"]
	in assoc r (map2 (fun x y -> (x, y)) groupe_du_cube lops)
;;

(*-------------------------------------------------------------------------------------------*)
(* 5 : GROUPE DES MOUVEMENTS *)
(*-------------------------------------------------------------------------------------------*)

(* groupe M des mouvements des minicubes *)

(* tri d'un mouvement selon l'ordre des indices *)
(*let trier mv1 = sort__sort (fun x y -> fst x < fst y) mv1;;*)

(* élément neutre de M *)
let e = map (fun x -> x, id) indices;;

(* conversion entre mouvement représenté par une fonction et mouvement *)
(* représenté par une liste : (int vect * int vect vect) list *)
let mv1_of_fun f =
	map (fun (x, y) -> (x, y /./ (f x))) e
;;
let fun_of_mv1 mv1 x =
	assoc x mv1
;;

(* mouvements globaux *)
let cst x = mv1_of_fun (fun t -> x);;

(* loi interne *)
(*
let prefix /*/ mv1 mv1' =
	let f = fun_of_mv1 mv1 and f' = fun_of_mv1 mv1'
	in
		let s t = t /:/ (f t)
		in mv1_of_fun (fun x -> (f x) /./ (f' (s x)))
;;
*)
let prefix /*/ mv mv' = map (fun i -> (i, let m = assoc i mv in let j = i /:/ m in (m /./ assoc j mv'))) indices;;

(* inverse d'un élément *)
let inverse mv1 = let l = map (fun (x, y) -> (x /:/ y, transpose y)) mv1
	in map (fun x -> x, assoc x l) indices
;;

(* mouvements de Rubik élémentaires *)

(* rotations dans le sens des aiguilles d'une montre d'un quart de tour de la *)
(* face - tranche interne dans le cas du cube 4x4 - normale au vecteur sortant 'v' *)
let rub v = mv1_of_fun
	(fun x -> if (x /|/ v) = 1 then rot v else id)
;;

(* mouvement inverse du précédent *)
let rub' v = inverse (rub v);;

(* enregistrement sur disque d'un mouvement: format portable *)
let enregistrer_mouv mv chemin =
	let rec aux mv =
		let traite x =
			printf__sprintf "%d%d%d" x.(0) x.(1) x.(2)
		in
			match mv with
				| [] -> ""
				| t :: r ->
							let (x, m) = t
							in
								traite x ^ traite m.(0) ^ traite m.(1) ^ traite m.(2);
								^ aux r
	in
		try
			let canalout = open_out chemin
			in
				output_string canalout (aux mv);
				close_out canalout
		with sys__Sys_error s -> failwith s
;;

(* lecture sur disque d'un mouvement : format portable *)
let couple_of_int_matrice s =
	let t = make_matrix 4 3 0 in
		for i = 0 to 3 do
			for j = 0 to 2 do
				t.(i).(j) <- s.(i * 3 + j)
			done
		done;
		(t.(0), [|t.(1); t.(2); t.(3)|])
;;
let int_vect s =
	let tete s =
		let l = string_length s in
			if l = 0 then ""
			else if s.[0] = `-` then sub_string s 0 2
			else sub_string s 0 1
	in
		let reste s =
			let l = string_length s
			and lt = string_length (tete s) in
				sub_string s lt (l - lt)
		in
			if s = "" then [||]
			else
				let rec aux ss =
					let t = tete ss and r = reste ss in
						if r <> "" then t :: aux r
						else [t]
				in vect_of_list (map int_of_string (aux s))
;;
let int_matrices_of_int_vect v =
	let lst = ref [] in
		for i = 0 to (vect_length v - 12) / 12 do
			lst := sub_vect v (12 * i) 12 :: !lst
		done;
		vect_of_list !lst
;;
let lire_mouv path =
	try
		let canalin = open_in path in
			let s = input_line canalin in
				close_in canalin;
				rev (list_of_vect (map_vect couple_of_int_matrice (int_matrices_of_int_vect (int_vect s))))
	with sys__Sys_error s1 -> print_string s1; e
;;

(*-------------------------------------------------------------------------------------------*)
(* 6 : TEST DE VALIDITÉ D'UN MOUVEMENT (APPARTENANCE AU SOUS-GROUPE DE RUBIK *)
(*-------------------------------------------------------------------------------------------*)

let marque x = [|0; 0; x.(2)|];;


(* morphisme 's: M -> S' et section 'l: S -> M' *)
(* construction d'une section 'l' de la suite exacte '0 -> K -> M -> S -> 0' *)
(* En Caml on représente la sujection 's' par 'sur', la section 'l' par 'sec' et 'gij' par 'gg i j' *)

(* éléments g_{ij} alias gg i j de G servant à construire cette section *)
let gg i j =
	let critere i j g = i /:/ g = j && marque i /:/ g = marque j
	in
		hd (select (critere i j) groupe_du_cube)
		(* cette liste devrait toujours contenir exactement un élément *)
;;

(* décomposition 'm = ker m /*/ sec (sur m)' d'un mouvement 'm' *)
(* avec 'ker m' élément du noyau de 'sur' *)
(* 'p' pour 'permutation': 'p = sur m' est la permutation 'p' des indices associée au mouvement 'm' *)
let sec p = mv1_of_fun (fun i -> gg i (p i));;
let sur m = fun i -> i /:/ fun_of_mv1 m i;;
let ker m = m /*/ inverse (sec (sur m));;

(* stabilisateur du coin d'indice i *)
let st i =
	let k = [|1; 1; 1|] and m = [|[|0; 0; 1|]; [|1; 0; 0|]; [|0; 1; 0|]|]
	in
		gg i k /./ m /./ gg k i
;;

(* liste des exposants des stabilisateurs des coins pour la partie noyau d'un mouvement 'm' *)
(* dans la décomposition précédente de 'm' *)    
let l_rtc m =
  let rtc_aux k = let f = fun_of_mv1 k in
      let indexc i = if f i = st i then 1
        else if f i = transpose (st i) then 2 else 0 in
        map indexc coins
  in rtc_aux (ker m)
;;

(* rotation totale des coins *)
let rtc m = it_list (prefix +) 0 (l_rtc m) mod 3;;

(* test d'appartenance d'un mouvement au sous-groupe R *)
let est_dans_R m = rtc m = 0;;

(* mouvement1 général défini par des permutations et rotations de coins *)
let nouveau_mv1 pc ec =
	let k = mv1_of_fun
		(fun i -> let x = ec i in
								if x = 0 then id
								else if x = 1 then st i
								else if x = 2 then transpose (st i)
								else failwith "mv1"
		)
	and l = mv1_of_fun (fun i -> gg i (pc i))
	in k /*/ l
;;


(*- mouvement aléatoire -*)

(* permutation aléatoire des coins *)
let pc_r () =
	let rlc = random_list coins in
		fun i -> assoc i (map2 (fun x y -> (x, y)) coins rlc)
;;

(* exposant aléatoire pour les coins *)
let ec_r = fun i -> random__int 3;;

(* mouvement aléatoire général *)
let mv1_r () = nouveau_mv1 (pc_r ()) ec_r;;

(* mouvement de Rubik aléatoire *)
let mv1_rubik_r () =
	(* rotation d'un seul coin i *)
	let rot_coin i n =
		nouveau_mv1 (fun x -> x) (fun j -> if j = i then n else 0)
	in
		let m = ref (mv1_r ()) in
			if rtc !m <> 0 then m := !m /*/ rot_coin [|1; 1; 1|] (3 - rtc !m);
			!m
;;

(*-------------------------------------------------------------------------------------------*)
(* 7 : MOUVEMENTS 2, ENCODAGE ET DÉCODAGE  *)
(*-------------------------------------------------------------------------------------------*)

(* le groupe des mouvements mv2 est isomorphe au groupe des mouvements mv1 *)
(* on utilise ici l'isomorphisme associé à la décomposition 
 'm = ker m /*/ sec (sur m)' d'un mouvement 'm' quelconque mv1 *)
 
type mv2 = {mutable rot_coins: int vect; mutable perm_coins: int vect};;

(* numérotation des coins *)
let v_coins = vect_of_list coins ;;
let num_of_coin c =
  let i = ref 0 in
    while c <> v_coins.(!i) do
      incr i
    done;
    !i
;;
let coin_of_num i = v_coins.(i);;

(*conversions des mouvements 'mv1' en mouvements 'mv2' *)
let mv2_of_mv1 mv1 =
  let rc = vect_of_list (l_rtc mv1)
  and pc = vect_of_list (map num_of_coin (map (sur mv1) (coins)))
  in
    {rot_coins = rc; perm_coins = pc}
;;

(* insère l'entier a à l'emplacement numéro i dans une copie de la permutation x et renvoie y *)
let insere a i x =
	let y = make_vect (vect_length x + 1) a in
		for j = 0 to i - 1 do
			y.(j) <- x.(j);
		done;
		for j = i + 1 to vect_length x do
			y.(j) <- x.(j - 1);
		done;
		y
;;

(* génération d'une liste des permutations des entiers de 0 à n - 1 *)
let rec permutations n =
	if n > 1 then
		let l = ref [] in let perm = permutations (n - 1) in for i = 0 to n - 1 do
					l := map (fun x -> insere (n - 1) i x) perm :: !l;
				done;
				flat_map (fun x -> x) !l
	else [[|0|]]
;;

(* les seules permutations des 8 numéros de coins qui nous intéressent sont celles qui laissent fixe 0 *)
(* i.e. le numéro du minicube d'indice [|1;1;1|] considéré ici comme fixe *)
let tab_perm_coins = vect_of_list (select (fun x -> x.(0) = 0) (permutations 8));;

(* récupération rapide du code d'une permutation i.e. de son indice dans 'tab_perm_coins' *)
(* un peu au détriment de la mémoire *)

let encode_perm_coins_par_table_intermediaire tab_perm_coins =
	let base8 c =
		c.(1) + (c.(2) lsl 3) + (c.(3) lsl 6) + (c.(4) lsl 9) + (c.(5) lsl 12) + (c.(6) lsl 15) + (c.(7) lsl 18)
	in
		let t = make_vect (1 + base8 [|0; 1; 2; 3; 4; 5; 6; 7|]) 0 in
			for i = 0 to vect_length tab_perm_coins - 1 do
				t.(base8 tab_perm_coins.(i)) <- i
			done;
			fun pc -> t.(base8 pc)
;;

let encode_perm_coins = encode_perm_coins_par_table_intermediaire tab_perm_coins;;

(* encodage des rotations des coins *)
(*
let encode_rot_coins rc =
	(* rc : en principe un tableau des 8 chiffres en base 3 d'un nombre entier 'num' multiple de 3
	et inférieur à 6561 = 9*9*9*9 i.e. num / 9 < 729 ; on choisit 'num' comme code de ce tableau *)
	(* le chiffre des unités est en premier, i.e. le premier élément de 'rc' à gauche est nul *)
	(* on calcule 'num' et on le divise par 9 car le deuxième élément est en principe opposé *)
	(* modulo 3 à la somme des 6 suivants pour rc = mv2.rot_coins où mv2 est un mouvement de Rubik *)
	let l = list_of_vect rc in (List.fold_right (fun x y -> x + 3 * y) l 0) / 9;;
*)

let tab_rot_coins =
	let chiffres num = (* nombres entiers 'num' multiple de 9 ayant 8 chiffres en base 3 *)
		let rc = make_vect 8 0 in
			let n = ref num in
				for i = 2 to 7 do
					rc.(i) <- !n mod 3;
					n := !n / 3
				done;
				rc.(1) <- (
				(*let x = it_list (prefix +) 0 (list_of_vect rc) mod 3*)
				(*let x = Array.fold_left ( + ) 0 rc mod 3*)
					let x = (rc.(2) + rc.(3) + rc.(4) + rc.(5) + rc.(6) + rc.(7)) mod 3
					in (* pour rc = mv2.rot_coins où mv2 est un mouvement de Rubik *)
						if x <> 0 then 3 - x else 0);
				rc
	in
		let t = make_vect 729 [|0; 0; 0; 0; 0; 0; 0; 0|] in
			for i = 0 to 728 do
				t.(i) <- chiffres i
			done;
			t
;;

(* récupération rapide du code du tableau des rotations des coins i.e. de son indice dans 'tab_rot_coins' *)

let encode_rot_coins_par_table_intermediaire tab_rot_coins =
	let base4 c = c.(2) + (c.(3) lsl 2) + (c.(4) lsl 4) + (c.(5) lsl 6) + (c.(6) lsl 8) + (c.(7) lsl 10)
	in
		let t = make_vect (1 + base4 [|0; 0; 2; 2; 2; 2; 2; 2|]) 0 in
			for i = 0 to vect_length tab_rot_coins - 1 do
				t.(base4 tab_rot_coins.(i)) <- i
			done;
			fun rc -> t.(base4 rc)
;;

let encode_rot_coins = encode_rot_coins_par_table_intermediaire tab_rot_coins;;

(* encodage des mouvements mv2  laissant fixe le minicube numéroté 0 *)
let encode2 mv2 = 729 * encode_perm_coins (mv2.perm_coins) + encode_rot_coins (mv2.rot_coins);;

(* décodage associé *)
let decode2 num =
	let (rc, pc) = (num mod 729,num / 729) in
		{rot_coins = tab_rot_coins.(rc); perm_coins = tab_perm_coins.(pc)}
;;

(* loi de groupe '$' pour les mouvements 'mv2' : correspond  par isomorphisme à '/*/' *)
let prefix $ m m' =
	let nc = vect_length m.rot_coins and npc = vect_length m.perm_coins
	in let rc = make_vect nc 0 and pc = make_vect npc 0
		in
			for i = 0 to nc - 1 do
				rc.(i) <- (m.rot_coins.(i) + m'.rot_coins.(m.perm_coins.(i))) mod 3
			done;
			for i = 0 to npc - 1 do
				pc.(i) <- m'.perm_coins.(m.perm_coins.(i))
			done;
			{rot_coins = rc; perm_coins = pc}
;;

(* élément neutre *)
let e2 = {
    rot_coins = [|0; 0; 0; 0; 0; 0; 0; 0|];
    perm_coins = [|0; 1; 2; 3; 4; 5; 6; 7|]}
;;

(* mouvements de Rubik élémentaires par quarts de tours des faces gauche, basse et postérieure *)
let rg = mv2_of_mv1 (rub [| 0; - 1; 0 |]);
and rg' = mv2_of_mv1 (rub' [| 0; - 1; 0 |]);
and rb = mv2_of_mv1 (rub [| 0; 0; - 1 |]);
and rb' = mv2_of_mv1 (rub' [| 0; 0; - 1 |])
and rp = mv2_of_mv1 (rub [| - 1; 0; 0 |]);
and rp' = mv2_of_mv1 (rub' [| - 1; 0; 0 |]);
;;
(* demi-tours correspondants *)
let dg = rg $ rg
and db = rb $ rb
and dp = rp $ rp
;;

(* listes de générateurs pour construire la table de transitions du groupe *)
let liste_gen_qdt = [rg'; rb'; rp'; rg; rb; rp];;
let liste_gen_dt_qdt = [dg; db; dp; rg'; rb'; rp'; rg; rb; rp];;

(* fonction abrégeant chaque générateur par un caractère dans la table de transitions *)
let chr (gen: mv2) =
	assoc gen [rg, `g`; rg', `G`; rb, `b`; rb', `B`; rp, `p`; rp', `P`; dg, `D`; db, `H`; dp, `A`];;

(* listes des caractères représentant les générateurs dans la table *)
(*let liste_chr_gen_qdt = [`g`; `G`; `b`; `B`; `p`; `P`];;
let liste_chr_gen_dt_qdt = [`D`; `H`; `A`; `g`; `G`; `b`; `B`; `p`; `P`];;*)

(* fonction donnant pour chaque caractère l'inverse du générateur représenté : *)
(* dans cette fonction `p` -> rp` et `P` -> rp etc... alors que dans la *)
(* fonction précédente rp -> `p` et rp` -> `P` etc... *)
let gen (chr: char) = match chr with
	| `g` -> rg' | `G` -> rg
	| `b` -> rb' | `B` -> rb
	| `p` -> rp' | `P` -> rp
	| `D` -> dg
	| `H` -> db
	| `A` -> dp
	| _ -> failwith "gen"
;; 

(* écriture dans le fichier 'nom' d'un tableau 'tab' de caractères *) 
let ecrire_table nom tab =
	let canal_out = open_out nom in
		for i = 0 to vect_length tab - 1 do
			printf__fprintf canal_out "%c" tab.(i);
		done;
		close_out canal_out
;;

(* lecture du fichier précédent *)
let lire_table nom =
	let canalin = open_in nom in
		let len = in_channel_length canalin
		in
		(*let bytes = Bytes.create len in*)
			let bytes = create_string len in
				really_input canalin bytes 0 len;
				close_in canalin;
				let v = make_vect len ` ` in
					for i = 0 to len - 1 do
						v.(i) <- bytes.[i]
					done;
					v
;;

(* construction  et enregistrement des tables de transitions *)
(* avec pour générateurs les quarts de tours *)
(* puis les demi-tours et quarts de tours *)
(* ou lecture des tables de transitions enregistrées *)
let tabchar nom message liste_gen = try
		lire_table nom
	with sys__Sys_error s -> (printf__printf message;
			print_newline ();
			let tc = transitions 3674160 e2 (prefix $) encode2 decode2 liste_gen chr in
				ecrire_table nom tc;
				tc)
;;
let tabchar_qdt = tabchar "tab_qdt" "Calcul d'une table de transitions par quarts de tours:" liste_gen_qdt;;
let tabchar_dt_qdt = tabchar "tab_dt_qdt" "Calcul d'une table de transitions par demi-tours et quarts de tours:" liste_gen_dt_qdt;;

(* retrouver à partir du code d'un mouvement une séquence minimale de Rubik résolvant ce mouvement *)
(* le code i donne en t.(i) une manoeuvre de Rubik qui ramène à un mouvement plus proche du neutre *)
(* on reprend avec le code j de ce mouvement plus proche et t.(j) etc... *)

let rec solution tabchar x =
	let c = tabchar.(x) in
		if c = `$` then ""
		else (
			let tr ch = match ch with
				| `p` -> "p'" | `g` -> "g'" | `b` -> "b'"
				| `P` -> "p" | `G` -> "g" | `B` -> "b"
				| `A` -> "p2" | `D` -> "g2" | `H` -> "b2" | _ -> make_string 1 ch
			in
				tr c ^ solution tabchar (encode2 (decode2 x $ gen c))
		)
;;

let solution_qdt = solution tabchar_qdt and solution_dt_qdt = solution tabchar_dt_qdt;;

(* application au mouvement du cube basculé *)
let solutions cube =
	let p = cst cube.context1.matrice and mv0 = cube.mouvement1.mv1 in
		let mv = inverse p /*/ mv0 /*/ p in
			let r = fun_of_mv1 mv [| 1; 1; 1 |] in
				if r = id then
					let code2 = encode2 (mv2_of_mv1 mv) in
						(solution_qdt code2, solution_dt_qdt code2)
				else ("", "")
;;

(*-------------------------------------------------------------------------------------------*)
(* 8 : GRAPHISME *)
(*-------------------------------------------------------------------------------------------*)

(* couleur rvb de la  couleur c *)
let couleur_rvb_de_couleur c =
	match c with
		| ROUGE -> graphics__red
		| ORANGE -> graphics__rgb 255 165 0
		| BLEU -> graphics__rgb 0 150 225
		| VERT -> graphics__green
		| JAUNE -> graphics__yellow
		| BLANC -> graphics__white
		| GRIS -> graphics__rgb 100 100 100
;;

(* association entre couleurs et vecteurs normaux aux faces du cube *)
let couleur_de_face v =
	match vect v with
		| 1, 0, 0 -> ORANGE
		| - 1, 0, 0 -> ROUGE
		| 0, 1, 0 -> VERT
		| 0, - 1, 0 -> BLEU
		| 0, 0, 1 -> BLANC
		| 0, 0, - 1 -> JAUNE
		| _ -> GRIS
;;

let couleur_rvb_de_face v =
	couleur_rvb_de_couleur (couleur_de_face v)
;;

let nom_couleur_de_face v =
	match vect v with
		| 1, 0, 0 -> "orange"
		| - 1, 0, 0 -> "rouge"
		| 0, 1, 0 -> "vert"
		| 0, - 1, 0 -> "bleu"
		| 0, 0, 1 -> "blanc"
		| 0, 0, - 1 -> "jaune"
		| _ -> "?"
;;

let nom_de_couleur couleur =
	match couleur with
		| ORANGE -> "ORANGE"
		| ROUGE -> "ROUGE"
		| VERT -> "VERT"
		| BLEU -> "BLEU"
		| BLANC -> "BLANC"
		| JAUNE -> "JAUNE"
		| _ -> "GRIS"
;;

let prj (ox, oy, ux, uy) v pt3 =
	let proj x y z =
		let c = sqrt 6. /. 2. in
			(c *. (y -. x) /. sqrt 2., c *. (-. (x +. y) +. 2. *. z) /. sqrt 6.)
	and (x, y, z) = vect (map_vect float_of_int pt3)
	in
		let (x1, y1, z1) =
			if v /|/ [| 1; 1; 1 |] = 1 then (x, y, z)
			else match vect v with
				| (_, 0, 0) -> (x -. 7., y, z)
				| (0, _, 0) -> (x, y -. 7., z)
				| _ -> (x, y, z -. 7.)
		in
			(
				int_of_float (float_of_int ox +. fst (proj x1 y1 z1) *. float_of_int ux),
					int_of_float (float_of_int oy +. snd (proj x1 y1 z1) *. float_of_int uy)
			)
;;

(* la fonction 'drawPoly' est utilisée pour tracer le pourtour des projections *)
(* des faces des minicubes *)
let drawPoly poly =
	let (x, y) = poly.(0) in graphics__moveto x y;
		for i = 1 to vect_length poly - 1 do
			let (x, y) = poly.(i) in graphics__lineto x y
		done;
		let (x, y) = poly.(0) in graphics__lineto x y;
;;

(* la fonction 'draw' est utilisée pour dessiner la projection 'x' d'une face *)
(* de minicube en superposant le tracé du pourtour à la couleur de remplissage *)
let draw x =
	let a, b = x in
		graphics__set_color b;
		graphics__fill_poly a;
		graphics__set_color graphics__black;
		drawPoly a
;;

(* 'face c v' renvoie, si le minicube à l'emplacement d'indice 'c' a une face F *)
(* dans la face du Rubik's cube normale au vecteur sortant 'v', un vecteur *)
(* à 4 composantes faisant le tour des 4 sommets de F *)

let coeff = ref 1;;

let face c v =
	let e = v /|/ [| 1; 1; 1 |] in let w = [| e; e; e |] in
			let w1 = w /:/ rot v in
				let w2 = w1 /:/ rot v in
					let w3 = w2 /:/ rot v in
						let l = [w; w1; w2; w3] in
							let add m = for i = 0 to 2 do m.(i) <- m.(i) + !coeff * c.(i) done
							in
								do_list add l;
								vect_of_list l;
;;

(* 'faces c' renvoie une liste de couples : la première composante est un vecteur listant les 3 vecteurs unitaires *)
(* sortants normaux aux faces visibles du minicube centré en 'c' *)
(* et la seconde est un vecteur dont la composante numéro i est un vecteur faisant le tour des 4 sommets de la face *)
(* visible du mincube normale au vecteur numéro i précédent *)

let faces c =
	let d = diag c.(0) c.(1) c.(2) in
				d, map_vect (face c) d
;;

(* 'affiche1' affiche le minicube centré en centre_adh déplacé par matrice_adh *)
(* avec centre_adh et matrice_adh exprimés en repère adh *)
(* mdp est la matrice de passage de ovb à adh : x_ovh = x_adh /:/ (transpose mdp) , x_adh = x_ovb /:/ mdp *) 
let affiche1 plan mdp (centre_adh,matrice_adh)=
		let d, f = faces centre_adh in
			for i = 0 to vect_length d - 1 do
				let v = d.(i) /:/ matrice_adh in
					let g = map_vect (fun x -> x /:/ matrice_adh) f.(i) in
						draw ((map_vect (prj plan v)
										g),
								couleur_rvb_de_face (d.(i) /:/ transpose mdp));
			done
;;

(* 'affiche_mouvement_aux' affiche le mouvement 'mv_adh' exprimé en repère 'adh' *) 
(* en appliquant 'affiche1' à chaque minicube déplacé par ce mouvement *)
let affiche_mouvement_aux plan mdp mv_adh =
		do_list (affiche1 plan mdp) mv_adh
;;

(* 'affiche_mouvement' affiche le mouvement 'mv_ovb' exprimé en repère 'ovb' *)
let affiche_mouvement plan mdp mv_ovb =
	let p = cst mdp in
		affiche_mouvement_aux plan mdp (inverse p /*/ mv_ovb /*/ p)
;;

(*-------------------------------------------------------------------------------------------*)
(* 9 : AFFICHAGE DE SOLUTIONS MINIMALES par quarts de tours ou demi-tours et quarts de tours *)
(*-------------------------------------------------------------------------------------------*)

let dessine_solution cube solution ory =
	let (ox, _, _, _) = cube.repere1.plan in let lb = 2 * ox / 18 in
			let width = 6 * lb in
				let p = make_vect 4 (0, 0) in
				let (largeur,hauteur)=graphics__text_size solution in
						let orx = 0 in
							p.(0) <- (orx, ory);
							p.(1) <- (orx + width, ory);
							p.(2) <- (orx + width, ory + 30);
							p.(3) <- (orx, ory + 30);
							draw (p, graphics__white);
							p.(1) <- (orx + largeur, ory);
							p.(2) <- (orx + largeur, ory + hauteur);
							p.(3) <- (orx, ory + hauteur);
							graphics__moveto (orx + width / 6) (ory + 10);
							graphics__set_color graphics__black;
							graphics__draw_string solution;
														
;;

let dessine_solutions cube solutions =
	let (solution_qdt, solution_qdt_dt) = solutions in
		dessine_solution cube solution_qdt 60;
		dessine_solution cube solution_qdt_dt 90;
;;

(* 'dessine_cube' affiche le cube 'cube' avec son mouvement actuel *)
(* et affiche des solutions minimales en nombre de quarts de tours *)
(* et en nombre de quarts de tours ou demi-tours des faces gauche, postérieure et basse *)
(* en considérant le minicube antérieur haut droit comme fixe (basculer éventuellement le cube) *)
(* (basculer éventuellement le cube) *)

let dessine_cube cube = affiche_mouvement cube.repere1.plan cube.context1.matrice cube.mouvement1.mv1;
	dessine_solutions cube (solutions cube);;

(*-------------------------------------------------------------------------------------------*)
(* 10 : RÉSOLUTION DU CUBE 2x2 par niveaux : "supérieur, inférieur" *)
(*-------------------------------------------------------------------------------------------*)

(* résolution du cube 2x2 *)
exception Orienter_les_coins;;
exception Descendre_coin;;
exception Remonter_coin;;
exception Placer_les_coins;;

let nom_position_de_face v =
	match vect v with
		| (1, 0, 0) -> "a"
		| (0, 1, 0) -> "d"
		| (0, 0, 1) -> "h"
		| (- 1, 0, 0) -> "p"
		| (0, - 1, 0) -> "g"
		| (0, 0, - 1) -> "b"
		| _ -> failwith "nom_position_de_face"
;;

let associe mat s =
	let adh = map_vect nom_position_de_face (mat)
	and pgb = map_vect nom_position_de_face ((- 1) /../ mat)
	in
		match s with
			| "a" -> adh.(0)
			| "d" -> adh.(1)
			| "h" -> adh.(2)
			| "p" -> pgb.(0)
			| "g" -> pgb.(1)
			| "b" -> pgb.(2)
			| "a'" -> adh.(0) ^ "'"
			| "d'" -> adh.(1) ^ "'"
			| "h'" -> adh.(2) ^ "'"
			| "p'" -> pgb.(0) ^ "'"
			| "g'" -> pgb.(1) ^ "'"
			| "b'" -> pgb.(2) ^ "'"
			| _ -> failwith "associe"
;;

(*-------------------------------------------------*)

(* en repère adh, la matrice de passage dans le groupe du cube telle que dans le repère adh associé *)
(* le coin centré en x dans l'état mv ait les couleurs adh coul1, coul2, coul3 *)
let context_adh_aux (coul1, coul2, coul3) mv xx =
	let couleurs_adh context mouvement x =
		let eclate x = [| [| x.(0); 0; 0 |]; [| 0; x.(1); 0 |]; [| 0; 0; x.(2) |] |] in
			let p = context.matrice in
				let m = eclate x /./ transpose p /./ (fun_of_mv1 (inverse mouvement)) (x /:/ transpose p) in
					map_vect couleur_de_face m
	in
		{matrice = hd (select (fun p -> couleurs_adh {matrice = p} mv xx = [| coul1; coul2; coul3 |]) groupe_du_cube)}
;;

let context_adh (coul1, coul2, coul3) mv = context_adh_aux (coul1, coul2, coul3) mv [| 1; 1; 1 |];;

(*-------------------------------------------------*)

(* les couleurs en repère adh des faces antérieure, droite et gauche du cube non mélangé *)
let couleurs_faces_adh cube =
	let _ = cube.context1.matrice in
		let feminin adjectif = match adjectif with
				| "blanc" -> "blanche"
				| "vert" -> "verte"
				| "bleu" -> "bleue"
				| _ -> adjectif
		in
			let noms = map_vect (fun x -> feminin (nom_couleur_de_face x)) (transpose cube.context1.matrice) in
				printf__sprintf "Résultat avec :\nface antérieure %s\nface droite %s \nface haute %s" noms.(0) noms.(1) noms.(2)
;;

(* les couleurs en repère adh des faces visibles du minicube centré en x *)
let couleurs_adh cube x =
	let eclate x = [| [| x.(0); 0; 0 |]; [| 0; x.(1); 0 |]; [| 0; 0; x.(2) |] |] in
		let p = cube.context1.matrice and mv1 = cube.mouvement1.mv1 in
			let m = eclate x /./ transpose p /./ (fun_of_mv1 (inverse mv1)) (x /:/ transpose p) in
				map_vect couleur_de_face m
;;

(* les couleurs en repère adh des faces visibles du minicube centré en [|1; 1; 1|] *)
(* dans l'ordre antérieure, droite, haute *)
let couleurs_coin_adh cube = couleurs_adh cube [| 1; 1; 1 |];;

(* impression des noms des couleurs en repère adh des faces visibles du minicube centré en [|1; 1; 1|] *)
(* dans l'ordre antérieure, droite, haute *)
let noms_couleurs_coin_adh cube =
	let noms = map_vect nom_de_couleur (couleurs_adh cube [| 1; 1; 1 |])
	in
		printf__sprintf "(%s, %s, %s)" noms.(0) noms.(1) noms.(2)
;;

(*-------------------------------------------------*)

let affiche_mvt repere context mv1 = affiche_mouvement repere.plan context.matrice mv1;;

let nouveau_cube mouvement context repere anime =
	let dessine () = if !anime then affiche_mvt repere context mouvement.mv1
	in
		let rotations_faces () =
			let fct x () =
				let t = x /:/ transpose context.matrice in
					mouvement.mv1 <- mouvement.mv1 /*/ rub t;
					dessine () 
			and fct' x () =
				let t = x /:/ transpose context.matrice in
					mouvement.mv1 <- mouvement.mv1 /*/ rub' t;
					dessine () 
			in
				let (a, d, h) = vect (map_vect fct id)
				and (a', d', h') = vect (map_vect fct' id)
				and (p, g, b) = vect (map_vect fct idm)
				and (p', g', b') = vect (map_vect fct' idm)
				in (OPS (a, d, h), OPS (a', d', h'), OPS (p, g, b), OPS (p', g', b'))
		and rotations_cube () =
			let rotate pp () = context.matrice <- context.matrice /./ pp;
				dessine () 
			in
				let (a, d, h) = vect (map_vect rotate (map_vect rot id))
				and (a', d', h') = vect (map_vect rotate (map_vect rot' id))
				in
					(OPS (a, d, h), OPS (a', d', h'))

		in
			{
				anime1 = anime;
				mouvement1 = mouvement;
				mvi = {mv1 = mv1_of_fun (fun x -> matrice_nulle)};
				context1 = context;
				repere1 = repere;
				rotations_cube1 = rotations_cube ();
				rotations_faces1 = rotations_faces ();
				boutons1 = make_vect 1 {titre = ""; orx = 0; ory = 0; largeur = 0;
							hauteur = 0; couleur = BLANC; action = (fun () -> ()); bas = false};
				boutons11 = make_vect 1 {titre = ""; orx = 0; ory = 0; largeur = 0;
							hauteur = 0; couleur = BLANC; action = (fun () -> ()); bas = false}
			}
;;

let resoudre_le_cube cube =
	(* (y, m) = pos0 x : le minicube d'indice x est à l'emplacement d'indice y et m est sa matrice *)
	(* de déplacement (telle que y=xm) (repère ADH) *)
	let nbqt = ref 0
	and lo = ref ""
	and matr = cube.context1.matrice
	and mctx = ref (id)
	in
		let pos0 cube =
			let pos mv1 x =
				let m = let p = cube.context1.matrice in fun_of_mv1 (inverse (cst p) /*/ mv1 /*/ cst p) (x)
				in
					x /:/ m, m
			in
				(fun x -> pos cube.mouvement1.mv1 x)
		and pos1 cube =
			let pos mv1 p x =
				let m = (fun_of_mv1 (inverse (cst p) /*/ mv1 /*/ cst p) (x))
				in
					x /:/ m, transpose m
			in
				(fun x -> pos (inverse cube.mouvement1.mv1) cube.context1.matrice x)
		and
		(OPS (a, d, h), OPS (a', d', h'), OPS (p, g, b), OPS (p', g', b')) = cube.rotations_faces1
		and
		(OPS (a0, d0, h0), OPS (a0', d0', h0')) = cube.rotations_cube1
		in
			let op_with_name s =
				let la = [("a", a); ("p", p); ("h", h); ("b", b); ("d", d); ("g", g);
						("a'", a'); ("p'", p'); ("h'", h'); ("b'", b'); ("d'", d'); ("g'", g');
						("a0", a0); ("d0", d0); ("h0", h0); ("a0'", a0'); ("d0'", d0'); ("h0'", h0');]
				in
					assoc s la
			in
				let exec1 str =
					let temp = cube.context1.matrice in
						let listop = op_names_from_string str in
							exe1 (map op_with_name listop);
							let temp1 = cube.context1.matrice in
								mctx := !mctx /./ transpose temp /./ temp1;
				and exec str =
					let listop = op_names_from_string str in
						nbqt := !nbqt + list_length listop;
						if !mctx <> id then
							(
								lo := !lo ^ format_string (decomposition (transpose !mctx));
								mctx := id
							);
						exe1 (map op_with_name listop);
						lo := !lo ^ (format_string str);
						let m = (transpose cube.context1.matrice) /./ matr in
							print_string (format_string (concat (map (associe m) listop)))
				in
					
					let niveau_superieur () =
						lo := !lo ^ "\n(* NIVEAU SUPÉRIEUR *)\n";
						let placer_et_orienter_les_coins () =
							let descendre_coin () =
								let w, m = pos0 cube [|1; 1; 1|] in
									if (w <> [|1; 1; 1|]) || (m <> id) then
										match vect w with
										| (- 1, 1, 1) -> exec "p'b'p"
										| (- 1, - 1, 1) -> exec "pbbp'"
										| (1, - 1, 1) -> exec "gbg'"
										| (1, 1, 1) -> exec "aba'b'"
										| (- 1, 1, - 1) -> exec "b'"
										| (- 1, - 1, - 1) -> exec "bb"
										| (1, - 1, - 1) -> exec "b"
										| (1, 1, - 1) -> ()
										| _ -> raise Descendre_coin
							and remonter_coin () =
								let (v, m) = pos0 cube [|1; 1; 1|] in
									if (v = [|1; 1; 1|]) && (m = id) then ()
									else
										let w = m.(2) in match vect w with
											| (1, 0, 0) -> exec "da'd'a"
											| (0, 1, 0) -> exec "a'dad'"
											| (0, 0, - 1) -> exec "ab'a'bbda'd'a"
											| _ -> raise Remonter_coin
							in
								let t = [|"antérieur droit"; "postérieur droit"; "postérieur gauche"; "antérieur gauche"|] in
									for i = 0 to 3 do
										printf__printf "(* - Mise en place du coin %s *)\n" t.(i);
										lo := !lo ^ printf__sprintf "(* - Mise en place du coin %s *)\n" t.(i);
										descendre_coin ();
										remonter_coin ();
										exec1 "h0"
									done;
						in
							placer_et_orienter_les_coins ()
					
					and niveau_inferieur () =
						lo := !lo ^ "\n(* NIVEAU INFÉRIEUR retourné *)\n";
						exec1 "d0d0";
						if !mctx <> id then
							(
								lo := !lo ^ format_string (decomposition (transpose !mctx));
								mctx := id
							);
						
						let placer_les_coins () =
							lo := !lo ^ "(* - Placer les coins *)\n";
							let permuter () =
								(* sens horaire *)
								exec "g'hdh'ghd'h'"
							and permuter' () =
								(* sens inverse *)
								exec "hdh'g'hd'h'g"
							in
								let chercher_un_coin_bien_place () =
									let i = ref 0 in
										while !i < 4 && fst (pos0 cube [|1; 1; 1|]) <> [|1; 1; 1|] do
											exec1 "h0";
											incr i
										done;
										!i
								in
									let j = chercher_un_coin_bien_place () in
										
										if j = 4 (* aucun coin bien placé *) then (
											(* permutation paire avec transposition : transposition de deux sommets consécutifs et des deux autres, ou demi-tour *)
												let v, _ = pos0 cube [|1; 1; 1|] in match vect v with
													| (- 1, 1, 1) -> exec "ghah'a'g'"
													| (1, - 1, 1) -> exec1 "h0'"; exec "ghah'a'g'"
													| (- 1, - 1, 1) -> exec "hh"
													| _ -> raise Placer_les_coins
											)
										else (
												let v, _ = pos0 cube [|1; - 1; 1|] in match vect v with
													| (- 1, - 1, 1) -> permuter' ()
													| (- 1, 1, 1) -> permuter ()
													| _ -> ()
											)
						
						and orienter_les_coins () =
							lo := !lo ^ "(* - Orienter les coins *)\n";
							let config () =
								(*
								let v1 = map_vect st [|[|1; 1; 1|]; [|- 1; 1; 1|]; [|- 1; - 1; 1|]; [|1; - 1; 1|]|] and
								v2 = map_vect snd (map_vect (pos1 cube) [|[|1; 1; 1|]; [|- 1; 1; 1|]; [|- 1; - 1; 1|]; [|1; - 1; 1|]|])
								in map_vect2 (fun x y -> (if y.(2) = [|0; 0; 1|] then 0 else if y.(2) /:/ x = [|0; 0; 1|] then 1 else - 1)) v1 v2
								*)
								let v1 = map st [[|1; 1; 1|]; [|- 1; 1; 1|]; [|- 1; - 1; 1|]; [|1; - 1; 1|]] and
								v2 = map snd (map (pos1 cube) [[|1; 1; 1|]; [|- 1; 1; 1|]; [|- 1; - 1; 1|]; [|1; - 1; 1|]])
								in vect_of_list (map2 (fun x y -> (if y.(2) = [|0; 0; 1|] then 0 else if y.(2) /:/ x = [|0; 0; 1|] then 1 else - 1)) v1 v2)
							and
							n_config conf =
								let i = ref 0 in for j = 0 to 3 do
										if conf.(j) = 0 then incr i
									done;
									!i
							in
								let rec aux () =
									let conf = config () in
									(*Printf.printf "[|%d; %d; %d; %d|]\n" conf.(0) conf.(1) conf.(2) conf.(3);*)
										let n = n_config conf in match n with
											| 0 -> if conf.(3) = 1 then exec "dhd'hdh'h'd'" else exec1 "h0"; aux ()
											| 1 -> (
													match (conf.(0), conf.(1), conf.(2), conf.(3)) with
													| (0, 1, 1, 1) -> exec1 "h0"; exec "dhd'hdh'h'd'"
													| (1, 0, 1, 1) -> exec1 "h0h0"; exec "dhd'hdh'h'd'"
													| (1, 1, 0, 1) -> exec1 "h0'"; exec "dhd'hdh'h'd'"
													| (1, 1, 1, 0) -> exec "dhd'hdh'h'd'"
													| (0, - 1, - 1, - 1) -> exec1 "h0'"; exec "dhhd'h'dh'd'"
													| (- 1, 0, - 1, - 1) -> exec "dhhd'h'dh'd'"
													| (- 1, - 1, 0, - 1) -> exec1 "h0"; exec "dhhd'h'dh'd'"
													| (- 1, - 1, - 1, 0) -> exec1 "h0h0"; exec "dhhd'h'dh'd'"
													| _ -> failwith "erreur dans orienter_les coins"
												);
												aux ()
											| 2 -> if conf.(3) = - 1 then exec "dhd'hdh'h'd'" else exec1 "h0"; aux ()
											| 4 -> ()
											| _ -> failwith "erreur 2 dans orienter_les coins"
								in
									aux ();
									if fst (pos0 cube [|1; 1; 1|]) <> [|1; 1; 1|] then exec "hh"
						in
						(* faire un quart de tour si la permutation des coins est impaire *)
						(* car 'placer les coins' suppose cette permutation paire *)
							if (sign coins (sur cube.mouvement1.mv1) = - 1) then
								(
									printf__printf "(* - Contrôle de parité *)\n";
									lo := !lo ^ "(* - Contrôle de parité *)\n";
									exec "h"
								)
							else ();
							
							print_string "(* - Placer les coins *)\n";
							placer_les_coins ();
							print_newline ();
							print_string "(* - Orienter les coins *)\n";
							orienter_les_coins ();
							print_newline ();
							let ss = decomposition (transpose (!mctx /./ transpose cube.context1.matrice /./ matr)) in
								if ss <> "" then
									(
										lo := !lo ^ "(* - Ramener le cube à sa position initiale *)\n";
										lo := !lo ^ format_string ss;
									);
					
					
					in
						try
							let mat = cube.context1.matrice
							and noms_couleurs = noms_couleurs_coin_adh cube
							in
								print_string "include \"exemples/Caml Light/Rubik/interfaces/interface2.ml\";;\n\n";
								printf__printf "let mv = lire_mouv \"mouv2222\";;\n";
								printf__printf "let ctx = context_adh %s mv;;\n" noms_couleurs;
								printf__printf "graphics__open_graph \" 612x612\";;\n\n";
								
								printf__printf "(* coin Antérieur Droit Haut : %s *)\n\n" noms_couleurs;
								
								print_string "(* RÉSOLUTION SANS ROTATIONS GLOBALES *)\n\n";
								printf__printf "(* %s *)\n\n" (couleurs_faces_adh cube);
								printf__printf "cube.context1.matrice <- ctx.matrice;;\n";
								printf__printf "cube.mouvement1.mv1 <- mv;;\n";
								printf__printf "dessine_cube ctx mv;;\n\n";
								
								printf__printf "\n(* NIVEAU SUPÉRIEUR *)\n";
								niveau_superieur ();
								printf__printf "\n(* NIVEAU INFÉRIEUR *)\n";
								niveau_inferieur ();
								
								cube.context1.matrice <- mat;
								dessine_cube cube;
								print_newline ();
								
								print_string "(* RÉSOLUTION AVEC ROTATIONS GLOBALES *)\n\n";
								printf__printf "cube.context1.matrice <- ctx.matrice;;\n";
								printf__printf "cube.mouvement1.mv1 <- mv;;\n";
								printf__printf "dessine_cube ctx mv;;\n\n";
								print_string !lo;
								
								print_newline ();
								!nbqt
						with
						| Orienter_les_coins ->
							print_string "erreur dans orienter_les_coins\n"; !nbqt
						| Placer_les_coins ->
							print_string "erreur dans placer_les_coins\n"; !nbqt
;;

(*-------------------------------------------------------------------------------------------*)
(* 11 : BOUTONS *)
(*-------------------------------------------------------------------------------------------*)
(* Gestion par boutons des mouvements globaux et des mouvements de Rubik *)

let bouton titre orx ory largeur hauteur couleur action =
	{titre = titre; orx = orx; ory = ory; hauteur = hauteur; largeur = largeur;
		couleur = couleur; action = action; bas = false}
;;

let inverse_bouton b =
	b.bas <- true;
	graphics__set_color graphics__black;
	graphics__fill_rect b.orx b.ory b.largeur b.hauteur;
	let (x, y) = graphics__text_size b.titre in
		graphics__moveto (b.orx + (b.largeur - x) / 2) (b.ory + (b.hauteur - y) / 2);
		graphics__set_color graphics__white;
		graphics__draw_string b.titre;
		let p = make_vect 4 (0, 0) in
			p.(0) <- (b.orx, b.ory);
			p.(1) <- (b.orx + b.largeur, b.ory);
			p.(2) <- (b.orx + b.largeur, b.ory + b.hauteur);
			p.(3) <- (b.orx, b.ory + b.hauteur);
			graphics__set_color graphics__black;
			drawPoly p
;;

let dessine_bouton b =
	b.bas <- false;
	graphics__set_color (couleur_rvb_de_couleur (b.couleur));
	graphics__fill_rect b.orx b.ory b.largeur b.hauteur;
	let (x, y) = graphics__text_size b.titre in
		graphics__moveto (b.orx + (b.largeur - x) / 2) (b.ory + (b.hauteur - y) / 2);
		graphics__set_color graphics__black;
		graphics__draw_string b.titre;
		let p = make_vect 4 (0, 0) in
			p.(0) <- (b.orx, b.ory);
			p.(1) <- (b.orx + b.largeur, b.ory);
			p.(2) <- (b.orx + b.largeur, b.ory + b.hauteur);
			p.(3) <- (b.orx, b.ory + b.hauteur);
			graphics__set_color graphics__black;
			drawPoly p
;;

let set_action bouton action =
	bouton.action <- action
;;

let gestion_bouton bouton mouse_down mousex mousey =
	if bouton.orx < mousex && mousex < bouton.orx + bouton.largeur
		&& bouton.ory < mousey && mousey < bouton.ory + bouton.hauteur then (
			if mouse_down then (
					if not bouton.bas then (
							inverse_bouton bouton;
						)
				)
			else (
					if bouton.bas then (
							dessine_bouton bouton;
							bouton.action ()
						)
				)
		)
	else (
			if bouton.bas then (
					dessine_bouton bouton
				)
		)
;;

(*-------------------------------------------------------------------------------------------*)
(* 12 : RANGÉES DE BOUTONS POUR LA MANIPULATION DU CUBE *)
(*-------------------------------------------------------------------------------------------*)

(* Gestion par boutons des mouvements globaux et des mouvements de Rubik *)
(* Pour fenêtre de largeur 612 et hauteur 612 *)
(* Cube avec en bas de fenêtre une rangée de 18 boutons de largeur 34 : 18 x 34 = 612 *)

(* Fenêtre de largeur 612 et hauteur 612 : origine  au centre (306,306), unités : 20,20 *)


(* Rangée de boutons en bas de fenêtre pour faire tourner le cube globalement *)
let dessine_boutons1 cube =
	let couleur_titre titre =
		let face titre = match titre with
				| "A" | "A'" | "a" | "a'" -> [|1; 0; 0|]
				| "D" | "D'" | "d" | "d'" -> [|0; 1; 0|]
				| "H" | "H'" | "h" | "h'" -> [|0; 0; 1|]
				| "p" | "p'" -> [|- 1; 0; 0|]
				| "g" | "g'" -> [|0; - 1; 0|]
				| "b" | "b'" -> [|0; 0; - 1|]
				| _ -> failwith "face"
		in
			couleur_de_face ((face titre) /:/ transpose cube.context1.matrice)
	in
		for i = 0 to vect_length cube.boutons1 - 1 do
			cube.boutons1.(i).couleur <- couleur_titre cube.boutons1.(i).titre;
			dessine_bouton cube.boutons1.(i)
		done
;;

let cree_boutons1 cube =
	let titres =
		[|"A"; "A'"; "H"; "H'"; "D"; "D'";
			"a"; "a'"; "h"; "h'"; "d"; "d'";
			"p"; "p'"; "b"; "b'"; "g"; "g'"
		|]
	in
		let set_actions boutons =
			let (OPS (a0, d0, h0), OPS (a0', d0', h0')) = cube.rotations_cube1
			and (OPS (a, d, h), OPS (a', d', h'), OPS (p, g, b), OPS (p', g', b')) =
				cube.rotations_faces1
			in
				let v = [|a0; a0'; h0; h0'; d0; d0'; a; a'; h; h'; d; d'; p; p'; b; b'; g; g'|]
				and op_names = [|"a0"; "a0'"; "h0"; "h0'"; "d0"; "d0'"; "a"; "a'"; "h"; "h'"; "d"; "d'"; "p"; "p'"; "b"; "b'"; "g"; "g'"|]
				in
					for i = 0 to 5 do
						set_action boutons.(i) (fun () -> v.(i) (); print_string (op_names.(i));
											dessine_cube cube;
											dessine_boutons1 cube;
											flush std_out)
					done;
					for i = 6 to vect_length v - 1 do
						set_action boutons.(i) (fun () -> v.(i) (); print_string (op_names.(i));
											dessine_cube cube; flush std_out)
					done
		in
			let n = vect_length titres in
				let (ox, _, _, _) = cube.repere1.plan in let lb = 2 * ox / 18 in
						let boutons = make_vect n (bouton "" 0 0 0 0 BLANC (fun () -> ())) in
							for i = 0 to n - 1 do
								boutons.(i) <- {titre = titres.(i); orx = i * lb; ory = 0; hauteur = 30;
									largeur = lb; couleur = BLANC; action = (fun () -> ()); bas = false}
							done;
							set_actions boutons;
							cube.boutons1 <- boutons;
;;


let gestion_boutons1 cube is_down mousex mousey =
	for i = 0 to vect_length cube.boutons1 - 1 do
		gestion_bouton cube.boutons1.(i) is_down mousex mousey
	done;
;;

(* Rangée de boutons supplémentaire pour faire tourner le cube en prolongeant son mouvement de façon globale *)
let cree_boutons11 cube =
	let titres =
		[|"A"; "A'"; "H"; "H'"; "D"; "D'"|]
	in
		let set_actions boutons =
			let ctx = cube.context1 in
				set_action boutons.(0) (fun () -> cube.mouvement1.mv1 <- cube.mouvement1.mv1 /*/ cst (rot ([|1; 0; 0|] /:/ transpose ctx.matrice));
									dessine_cube cube; flush std_out);
				set_action boutons.(1) (fun () -> cube.mouvement1.mv1 <- cube.mouvement1.mv1 /*/ cst (rot' ([|1; 0; 0|] /:/ transpose ctx.matrice));
									dessine_cube cube; flush std_out);
				set_action boutons.(2) (fun () -> cube.mouvement1.mv1 <- cube.mouvement1.mv1 /*/ cst (rot ([|0; 0; 1|] /:/ transpose ctx.matrice));
									dessine_cube cube; flush std_out);
				set_action boutons.(3) (fun () -> cube.mouvement1.mv1 <- cube.mouvement1.mv1 /*/ cst (rot' ([|0; 0; 1|] /:/ transpose ctx.matrice));
									dessine_cube cube; flush std_out);
				set_action boutons.(4) (fun () -> cube.mouvement1.mv1 <- cube.mouvement1.mv1 /*/ cst (rot ([|0; 1; 0|] /:/ transpose ctx.matrice));
									dessine_cube cube; flush std_out);
				set_action boutons.(5) (fun () -> cube.mouvement1.mv1 <- cube.mouvement1.mv1 /*/ cst (rot' ([|0; 1; 0|] /:/ transpose ctx.matrice));
									dessine_cube cube; flush std_out);
		in
			let n = vect_length titres in
				let (ox, _, _, _) = cube.repere1.plan in let lb = 2 * ox / 18 in
						let boutons = make_vect n (bouton "" 0 0 0 0 BLANC (fun () -> ())) in
							for i = 0 to n - 1 do
								boutons.(i) <- {titre = titres.(i); orx = i * lb; ory = 30; hauteur = 30;
									largeur = lb; couleur = BLANC; action = (fun () -> ()); bas = false}
							done;
							set_actions boutons;
							cube.boutons11 <- boutons;
;;

let dessine_boutons11 cube =
	for i = 0 to vect_length cube.boutons11 - 1 do
		cube.boutons11.(i).couleur <- BLANC;
		dessine_bouton cube.boutons11.(i)
	done
;;

let gestion_boutons11 cube is_down mousex mousey =
	for i = 0 to vect_length cube.boutons11 - 1 do
		gestion_bouton cube.boutons11.(i) is_down mousex mousey
	done;
;;

exception Quitter;;

(* boutons divers *)
let cree_boutons cube actions =
	let largeur, hauteur = graphics__text_size "Résoudre"
	and largeur1, hauteur1 = graphics__text_size "Quitter"
	and largeur4, hauteur4 = graphics__text_size "Mélanger"
	and largeur2, hauteur2 = graphics__text_size "Restituer"
	and largeur5, hauteur5 = graphics__text_size "Composer"
	and largeur6, hauteur6 = graphics__text_size "Basculer"
	and _ = cree_boutons1 cube;
	and _ = cree_boutons11 cube
	in
		let
		bouton_quitter = bouton "Quitter" (graphics__size_x () - largeur1 - 20) (graphics__size_y () - hauteur1 - 20)
			(largeur1 + 10) (hauteur1 + 10) JAUNE (fun () -> raise Quitter)
		and bouton_resoudre = bouton "Résoudre" 20 (graphics__size_y () - hauteur - 20)
			(largeur + 10) (hauteur + 10) JAUNE actions.(0)
		and bouton_melanger = bouton "Mélanger" ((graphics__size_x () - largeur4) / 2 - 10) (graphics__size_y () - hauteur4 - 20)
			(largeur4 + 10) (hauteur4 + 10) JAUNE actions.(1)
		and bouton_restaurer = bouton "Restituer" ((graphics__size_x () - largeur2) / 2 - 10) (graphics__size_y () - 2 * (hauteur2 + 20))
			(largeur2 + 10) (hauteur2 + 10) JAUNE actions.(2)
		and bouton_composer = bouton "Composer" ((graphics__size_x () - largeur5) / 2 - 10) (graphics__size_y () - 3 * (hauteur5 + 20))
			(largeur5 + 10) (hauteur5 + 10) JAUNE actions.(3)
		and bouton_tourner = bouton "Basculer" ((graphics__size_x () - largeur6) / 2 - 10) (graphics__size_y () - 4 * (hauteur6 + 20))
			(largeur6 + 10) (hauteur6 + 10) JAUNE actions.(4)
		in
			[bouton_quitter; bouton_resoudre; bouton_melanger; bouton_restaurer; bouton_composer; bouton_tourner]
;;

let dessine_boutons liste_boutons =
	do_list dessine_bouton liste_boutons
;;

let gestion_boutons liste_boutons is_down mousex mousey =
	do_list (fun b -> gestion_bouton b is_down mousex mousey) liste_boutons
;;

(*-------------------------------------------------------------------------------------------*)
(* 13 : SAISIE D'UN MOUVEMENT DU CUBE *)
(*-------------------------------------------------------------------------------------------*)

exception Fin_de_recherche;;(* on a cliqué sur une face de minicube dans le panneau gauche *)

(* affichage en gris d'un minicube *)
let griser plan centre =
	let d, f = faces centre in
		for i = 0 to vect_length d - 1 do
			let v = d.(i) in
				draw ((map_vect (prj plan v)
						f.(i)),
					couleur_rvb_de_couleur GRIS)
		done
;;

let est_vide_a_gauche cube v = fun_of_mv1 cube.mvi.mv1 v <> matrice_nulle;;

let translate (ox, oy, ux, uy) h = (ox + h, oy, ux, uy);;

let affiche_mvi cube plan largeur =
	let plan1 = translate plan (largeur / 2) in
		do_list (griser plan1) indices;
		let p = cube.context1.matrice in
			let f i =
				if est_vide_a_gauche cube i then griser plan (i /:/ p)
				else (affiche1 plan p ((i /:/ p), id));
				if fun_of_mv1 cube.mvi.mv1 i <> matrice_nulle then
					affiche1 plan1 p ((i /:/ p), (transpose p /./ fun_of_mv1 cube.mvi.mv1 i /./ p))
			in
				do_list f indices
;;

(* pour les clics dans les faces des minicubes *)
(* est_dans_poly p x renvoie 'true' ssi le point 'x' est intérieur au quadrilatère convexe 'p'*)
let est_dans_poly p x =
	let prefix /-/ (a1, b1) (a2, b2) = (a1 - a2, b1 - b2)
	and det (a1, b1) (a2, b2) = a1 * b2 - a2 * b1
	in
		let p0 = p.(0) /-/ x
		and p1 = p.(1) /-/ x
		and p2 = p.(2) /-/ x
		and p3 = p.(3) /-/ x
		in
			det p0 p1 * det p1 p2 > 0
			&& det p1 p2 * det p2 p3 > 0
			&& det p2 p3 * det p3 p0 > 0
;;

let gestion_gauche_droite cube largeur =
	let choix_a_gauche = ref NIL and choix_a_droite = ref NIL
	and action plan mousex mousey =
		let x = ref NIL in
			try
				do_list (
					fun i ->
									let (d, f) = faces i in
										for j = 0 to 2 do
											let face = map_vect (prj plan d.(j)) f.(j) in
												if est_dans_poly face (mousex, mousey) then (x := COUPLE (i, d.(j)); raise Fin_de_recherche)
										done
				)
				indices;
				NIL
			with Fin_de_recherche -> !x
	in
		let gestion_a_gauche plan mousex mousey = (
				if mousex < largeur / 2 then (
						choix_a_gauche := action plan mousex mousey;
					)
			)
		and gestion_a_droite plan mousex mousey = (
				if mousex >= largeur / 2 then
					(
						choix_a_droite := action (translate plan (largeur / 2)) mousex mousey;
						let p = cube.context1.matrice in
							match !choix_a_droite with
								| COUPLE (w1, w2) ->
											(
												let est_vide_a_droite v = for_all (fun i -> i /:/ fun_of_mv1 cube.mvi.mv1 i <> v) indices
												in
													if est_vide_a_droite (w1 /:/ transpose p) then
														(
															(match !choix_a_gauche with
																	| COUPLE (v1, v2) ->
																				let mat = hd (select (fun m -> v1 /:/ m = w1 && v2 /:/ m = w2) groupe_du_cube)
																				in
																					(let f i = if i /:/ p <> v1 then fun_of_mv1 cube.mvi.mv1 i else p /./ mat /./ transpose p in
																							cube.mvi.mv1 <- map (fun i -> (i, f i)) indices
																					);
																					choix_a_gauche := NIL
																	| NIL -> ()
															);
														)
													else
														(
															cube.mvi.mv1 <- map (fun i -> (i, let m = fun_of_mv1 cube.mvi.mv1 i in if i /:/ m /:/ p = w1 then matrice_nulle else m)) indices;
															choix_a_gauche := NIL;
														)
											)
								| NIL -> ()
					);
			)
		in
			(gestion_a_gauche, gestion_a_droite)
;;

(*-------------------------------------------------------------------------------------------*)
(* 14 : BOUCLE DE SAISIE DU MOUVEMENT *)
(*-------------------------------------------------------------------------------------------*)

let tout_vide_a_gauche cube = for_all (est_vide_a_gauche cube) indices;;

let boucle_saisie cube s =
	graphics__open_graph s;
	let (largeur, hauteur) = (graphics__size_x ()), (graphics__size_y ())
	in
		let plan = (largeur / 4, hauteur / 2, largeur / 45, largeur / 45)
		in let (gestion_a_gauche, gestion_a_droite) = gestion_gauche_droite cube largeur
			in
				graphics__set_window_title "Composer un cube";
				affiche_mvi cube plan largeur;
				graphics__set_color graphics__black;
				graphics__moveto (largeur / 2) 0;
				graphics__lineto (largeur / 2) hauteur; (* cloison entre les deux panneaux *)
				printf__printf "\n-----------------------------------------------------------\n";
				printf__printf "COMPOSITION D'UN MOUVEMENT:\n\n";
				printf__printf "- Pour transférer un minicube de gauche à droite, cliquer sur une de ses faces puis cliquer à droite sur la destination de cette face (une face grise).\n";
				printf__printf "- Pour ramener un minicube de droite à sa place à gauche, cliquer sur une de ses faces.\n\n";
				printf__printf "- La rotation totale des minicubes est vérifiée quand tous les minicubes sont à droite.\n";
				printf__printf "- Les mouvements du groupe du Rubik's cube 2x2x2 sont ceux dont la rotation totale est nulle.";
				print_newline ();
				try
					let largeur1, hauteur1 = graphics__text_size "Quitter"
					in
						let bouton_quitter = bouton "Quitter" (graphics__size_x () - largeur1 - 20) (graphics__size_y () - hauteur1 - 20)
							(largeur1 + 10) (hauteur1 + 10) JAUNE (fun () -> raise Quitter)
						in
							dessine_bouton bouton_quitter;
							while true do
								(let status = graphics__wait_next_event [graphics__Button_down] in
										let mousex = status.graphics__mouse_x and mousey = status.graphics__mouse_y
										in gestion_bouton bouton_quitter true mousex mousey
								);
								let status = graphics__wait_next_event [graphics__Button_up] in
									let mousex = status.graphics__mouse_x and mousey = status.graphics__mouse_y
									in
										gestion_bouton bouton_quitter false mousex mousey;
										gestion_a_gauche plan mousex mousey;
										gestion_a_droite plan mousex mousey;
										affiche_mvi cube plan largeur;
										if tout_vide_a_gauche cube then (
												let rt = rtc cube.mvi.mv1 in
													printf__printf "rotation totale : %d" rt;
													print_newline ();
											);
							done;
				with Quitter ->
								graphics__close_graph ();
								if tout_vide_a_gauche cube then (
										cube.mouvement1.mv1 <- cube.mvi.mv1;
									)
								else (
										printf__printf "composition de mouvement inachevée";
										print_newline ();
									)
;;

(*-------------------------------------------------------------------------------------------*)
(* 15 : BOUCLE DE MANIPULATION DU CUBE 2x2 *)
(*-------------------------------------------------------------------------------------------*)

let boucle1 cube actions =
	try
		dessine_cube cube;
		let liste_boutons = cree_boutons cube actions
		in
			dessine_boutons liste_boutons;
			dessine_boutons1 cube;
			dessine_boutons11 cube;
			while true do
				let status = graphics__wait_next_event [graphics__Button_down; graphics__Button_up; graphics__Mouse_motion]
				in
					let mousex = status.graphics__mouse_x
					and mousey = status.graphics__mouse_y
					and is_down = status.graphics__button
					in
						gestion_boutons liste_boutons is_down mousex mousey;
						gestion_boutons1 cube is_down mousex mousey;
						gestion_boutons11 cube is_down mousex mousey;
			done;
	with Quitter -> graphics__close_graph ()
;;

(*-------------------------------------------------------------------------------------------*)
(* 16 : PROGRAMME PRINCIPAL *)
(*-------------------------------------------------------------------------------------------*)

let rec boucle cube =
	(let (ox, oy, _, _) = cube.repere1.plan in
			let (sx, sy) = (string_of_int (2 * ox), string_of_int (2 * oy)) in
				graphics__open_graph (" " ^ sx ^ "x" ^ sy);
				graphics__set_window_title "Résoudre le cube"
	);
	try
		boucle1 cube
		[|(* résoudre *)
			(fun () ->
								print_string "\n(*-----------------------------------------------------------*)\n";
								print_newline ();
								enregistrer_mouv cube.mouvement1.mv1 "mouv2222";
								let l = resoudre_le_cube cube in
									dessine_cube cube;
									printf__printf "\n(* nombre de quarts de tours : %d *)\n" l;
									print_newline ()
			);
			(* mélanger *)
			(fun () -> random__init (int_of_float (10000. *. sys__time ()));
								cube.mouvement1.mv1 <- mv1_rubik_r ();
								dessine_cube cube;
			);
			(* restituer *)
			(fun () -> cube.mouvement1.mv1 <- lire_mouv "mouv2222";
								dessine_cube cube;
			);
			(* composer *)
			(fun () -> let (largeur, hauteur) = (900, 500)
								in
									let l = string_of_int largeur and h = string_of_int hauteur in
										graphics__close_graph ();
										boucle_saisie cube (" " ^ l ^ "x" ^ h);
										boucle cube
			);
			(* basculer *)
			(fun () -> let p = cube.context1.matrice and mv = cube.mouvement1.mv1 in
									let x = [|1; 1; 1|] /:/ transpose p in
										let r = (fun_of_mv1 mv) x in
											cube.mouvement1.mv1 <- cube.mouvement1.mv1 /*/ cst (transpose r);
											dessine_cube cube;
			);
		|]
	with graphics__Graphic_failure s -> ()
;;

let mouvement = {mv1 = lire_mouv "mouv2222"};;
let context = {matrice = id};;
let repere = {plan = (306, 306, 20, 20)};;
let anime = ref true;;
let cube = nouveau_cube mouvement context repere anime;;
boucle cube;;







